<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User\Follow;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Dtos\Common\NotificationsCountDTO;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowEnum;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowModel;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowFinder;

final class UserFollowContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    /*
    |--------------------------------------------------------------------------
    | user relationship processor
    |--------------------------------------------------------------------------
    */

    public static function userRelationships(string $feedType, string $loadType): void
    {
        self::feedHelperInit($feedType, $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userIdFromReq = self::$request -> findKey(UserTable::ID, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

            $offset = self::$request -> findKeyOffset(UserFollowTable::ID, UserFollowTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | check if target user exists
            |--------------------------------------------------------------------------
            */

            $targetUserModel = UserModel::findFromId_throwException($userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserAvailable($targetUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::PRIVATE_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $followTableQuery = (new Query()) -> from(UserFollowTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | select where user id is matched
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingFollowers())
            {
                $followTableQuery -> where(UserFollowTable::FOLLOWED_USER_ID, $userIdFromReq);
            }
            else
            {
                $followTableQuery -> where(UserFollowTable::FOLLOWED_BY_USER_ID, $userIdFromReq);
            }

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $followTableQuery -> greaterThan(UserFollowTable::ID, $offset);
            }
            else
            {
                $followTableQuery -> lessThan(UserFollowTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $followTableQuery
                -> orderByDesc(UserFollowTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_USER_FOLLOW));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $followBeans = $followTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($followBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | else add to response
            |--------------------------------------------------------------------------
            */

            self::processBeans(UserFollowTable::getTableName(), $followBeans, function (UserFollowModel $model)
            {
                /*
                |--------------------------------------------------------------------------
                | list of users(who's models are selected)
                |--------------------------------------------------------------------------
                */

                $userId = self::isLoadingFollowers() ? $model -> getFollowedByUserId() : $model -> getFollowedUserId();

                self::addDependency(UserTable::getTableName(), $userId);
            });

            /*
            |--------------------------------------------------------------------------
            | process dependencies
            |--------------------------------------------------------------------------
            */

            self::processDependencies();

            /*
            |--------------------------------------------------------------------------
            | additional data | follow maps
            | -------------------------------------------------------------------------
            | help build follow button next to user tile
            |--------------------------------------------------------------------------
            */

            $usersContainer = self::$dataDock -> getContainer(UserTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                UserFollowTable::getTableName(),
                array(
                    UserFollowTable::FOLLOWED_USER_ID    => $usersContainer -> getIds(),
                    UserFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );
        });
    }

    public static function pendingRequestsCount(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            |  ensure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | initialize
            |--------------------------------------------------------------------------
            */

            $pendingRequestsCount = 0;

            /*
            |--------------------------------------------------------------------------
            | find unread notifications
            |--------------------------------------------------------------------------
            */

            $pendingRequestsCount = (new UserFollowFinder())
                -> setFollowedUserId(self::$authedUserModel -> getId())
                -> setMetaIsPending(UserFollowEnum::META_IS_PENDING_YES)
                -> findCount();

            /*
            |--------------------------------------------------------------------------
            | prepare a notification count dto
            |--------------------------------------------------------------------------
            */

            $notificationsCountDTO = (new NotificationsCountDTO())
                -> setLikeCount('0')
                -> setCommentCount('0')
                -> setFollowCount((string) $pendingRequestsCount)
                -> setOtherCount('0');

            /*
            |--------------------------------------------------------------------------
            | add to response
            |--------------------------------------------------------------------------
            */

            self::$response -> setContent(
                NotificationsCountDTO::DTO_NAME,
                array($notificationsCountDTO -> jsonSerialize())
            );

            /*
            |--------------------------------------------------------------------------
            | find the most recent pending follow request, and add user model to response
            |--------------------------------------------------------------------------
            */

            if ((int) $pendingRequestsCount > 0)
            {
                $finder = (new UserFollowFinder())
                    -> setFollowedUserId(self::$authedUserModel -> getId())
                    -> setMetaIsPending(UserFollowEnum::META_IS_PENDING_YES)
                    -> limit('1')
                    -> find();

                if ($finder -> isFound())
                {
                    $userFollowModel = $finder -> popModelFromResults();

                    $recentUserModel = UserModel::findFromId_noException($userFollowModel -> getFollowedByUserId());

                    if ($recentUserModel -> isModel())
                    {
                        self::addToResponse(UserTable::getTableName(), $recentUserModel -> getDataMap());
                    }
                }
            }
        });
    }

    /*
    |--------------------------------------------------------------------------
    | pending requests
    |--------------------------------------------------------------------------
    */

    public static function pendingRequests(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $offset = self::$request -> findKeyOffset(UserFollowTable::ID, UserFollowTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | finder
            |--------------------------------------------------------------------------
            */

            $followTableQuery = (new Query()) -> from(UserFollowTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | only pending requests
            |--------------------------------------------------------------------------
            */

            $followTableQuery -> where(UserFollowTable::META_IS_PENDING, UserFollowEnum::META_IS_PENDING_YES);

            /*
            |--------------------------------------------------------------------------
            | select for current user
            |--------------------------------------------------------------------------
            */

            $followTableQuery -> where(UserFollowTable::FOLLOWED_USER_ID, self::$authedUserModel -> getId());

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $followTableQuery -> greaterThan(UserFollowTable::ID, $offset);
            }
            else
            {
                $followTableQuery -> lessThan(UserFollowTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $followTableQuery
                -> orderByDesc(UserFollowTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_USER_FOLLOW));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $followBeans = $followTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($followBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | else add to response
            |--------------------------------------------------------------------------
            */

            self::processBeans(UserFollowTable::getTableName(), $followBeans, function (UserFollowModel $model)
            {
                /*
                |--------------------------------------------------------------------------
                | list of users(who's models are selected)
                |--------------------------------------------------------------------------
                */

                self::addDependency(UserTable::getTableName(), $model -> getFollowedByUserId());
            });

            /*
            |--------------------------------------------------------------------------
            | process dependencies
            |--------------------------------------------------------------------------
            */

            self::processDependencies();

            /*
            |--------------------------------------------------------------------------
            | additional data | follow maps
            | -------------------------------------------------------------------------
            | help build follow button next to user tile
            |--------------------------------------------------------------------------
            */

            $usersContainer = self::$dataDock -> getContainer(UserTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                UserFollowTable::getTableName(),
                array(
                    UserFollowTable::FOLLOWED_USER_ID    => $usersContainer -> getIds(),
                    UserFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );
        });
    }
}
